# SPDX-FileCopyrightText: 2018 Dean Miller for Adafruit Industries
#
# SPDX-License-Identifier: MIT

"""
`adafruit_epd.mcp_sram` - Adafruit MCP SRAM - sram driver
====================================================================================
CircuitPython driver for Microchip SRAM chips
* Author(s): Dean Miller
"""

from adafruit_bus_device import spi_device
from micropython import const

try:
    """Needed for type annotations"""
    from typing import Any, List

    from busio import SPI
    from digitalio import DigitalInOut
except ImportError:
    pass

__version__ = "2.15.0"
__repo__ = "https://github.com/adafruit/Adafruit_CircuitPython_EPD.git"

SRAM_SEQUENTIAL_MODE = const(1 << 6)


class Adafruit_MCP_SRAM_View:
    """An interface class that turns an SRAM chip into something like a memoryview"""

    def __init__(self, sram: int, offset: int) -> None:
        self._sram = sram
        self._offset = offset
        self._buf = [0]

    def __getitem__(self, i: int) -> Any:
        return self._sram.read(self._offset + i, 1)[0]

    def __setitem__(self, i: int, val: Any) -> None:
        self._buf[0] = val
        self._sram.write(self._offset + i, self._buf)


class Adafruit_MCP_SRAM:
    """supporting class for communicating with
    Microchip SRAM chips"""

    SRAM_READ = 0x03
    SRAM_WRITE = 0x02
    SRAM_RDSR = 0x05
    SRAM_WRSR = 0x01

    def __init__(self, cs_pin: DigitalInOut, spi: SPI):
        # Handle hardware SPI
        self._spi = spi_device.SPIDevice(spi, cs_pin, baudrate=8000000)
        self.spi_device = spi
        self.cs_pin = cs_pin
        self._buf = bytearray(3)
        self._buf[0] = Adafruit_MCP_SRAM.SRAM_WRSR
        self._buf[1] = 0x43
        with self._spi as spidev:
            spidev.write(self._buf, end=2)

    def get_view(self, offset: int) -> Adafruit_MCP_SRAM_View:
        """Create an object that can be used as a memoryview, with a given offset"""
        return Adafruit_MCP_SRAM_View(self, offset)

    def write(self, addr: int, buf: List, reg=SRAM_WRITE):
        """write the passed buffer to the passed address"""
        self._buf[0] = reg
        self._buf[1] = (addr >> 8) & 0xFF
        self._buf[2] = addr & 0xFF

        with self._spi as spi:
            spi.write(self._buf, end=3)
            spi.write(bytearray(buf))

    def read(self, addr: int, length: int, reg: int = SRAM_READ):
        """read passed number of bytes at the passed address"""
        self._buf[0] = reg
        self._buf[1] = (addr >> 8) & 0xFF
        self._buf[2] = addr & 0xFF

        buf = bytearray(length)
        with self._spi as spi:
            spi.write(self._buf, end=3)
            spi.readinto(buf)
        return buf

    def read8(self, addr: int, reg: int = SRAM_READ):
        """read a single byte at the passed address"""
        return self.read(addr, 1, reg)[0]

    def read16(self, addr: int, reg: int = SRAM_READ):
        """read 2 bytes at the passed address"""
        buf = self.read(addr, 2, reg)
        return buf[0] << 8 | buf[1]

    def write8(self, addr: int, value: int, reg: int = SRAM_WRITE):
        """write a single byte at the passed address"""
        self.write(addr, [value], reg)

    def write16(self, addr: int, value: int, reg: int = SRAM_WRITE):
        """write 2 bytes at the passed address"""
        self.write(addr, [value >> 8, value], reg)

    def erase(self, addr: int, length: int, value: Any):
        """erase the passed number of bytes starting at the passed address"""
        self._buf[0] = Adafruit_MCP_SRAM.SRAM_WRITE
        self._buf[1] = (addr >> 8) & 0xFF
        self._buf[2] = addr & 0xFF
        fill = bytearray([value])
        with self._spi as spi:
            spi.write(self._buf, end=3)
            for _ in range(length):
                spi.write(fill)
