# SPDX-FileCopyrightText: 2017 Scott Shawcroft, written for Adafruit Industries
# SPDX-FileCopyrightText: Copyright (c) 2020 Melissa LeBlanc-Williams for Adafruit Industries
#
# SPDX-License-Identifier: MIT
"""
`adafruit_portalbase`
================================================================================

Base Library for the Portal-style libraries.


* Author(s): Melissa LeBlanc-Williams

Implementation Notes
--------------------

**Software and Dependencies:**

* Adafruit CircuitPython firmware for the supported boards:
  https://github.com/adafruit/circuitpython/releases

"""

import gc
import time

import terminalio
from adafruit_bitmap_font import bitmap_font
from adafruit_display_text import wrap_text_to_lines
from adafruit_display_text.bitmap_label import Label
from adafruit_display_text.outlined_label import OutlinedLabel

__version__ = "3.3.0"
__repo__ = "https://github.com/adafruit/Adafruit_CircuitPython_PortalBase.git"


class PortalBase:
    """Class representing the Adafruit MagTag.

    :param network: An initialized network class instance.
    :param graphics: An initialized graphics class instance.
    :param url: The URL of your data source. Defaults to ``None``.
    :param headers: The headers for authentication, typically used by Azure API's.
    :param json_path: The list of json traversal to get data out of. Can be list of lists for
                    multiple data points. Defaults to ``None`` to not use json.
    :param regexp_path: The list of regexp strings to get data out (use a single regexp group).
                        Can be list of regexps for multiple data points. Defaults to ``None``
                        to not use regexp.
    :param default_bg: The path to your default background image file or a hex color.
                    Defaults to 0x000000.
    :param status_neopixel: The pin for the status NeoPixel. Use ``board.NEOPIXEL`` for the
                            on-board NeoPixel. Defaults to ``None``, to not use the status LED
    :param json_transform: A function or a list of functions to call with the parsed JSON.
                        Changes and additions are permitted for the ``dict`` object.
    :param success_callback: A function we'll call if you like, when we fetch data successfully.
                             Defaults to ``None``.
    :param debug: Turn on debug print outs. Defaults to False.

    """

    def __init__(  # noqa: PLR0912,PLR0913 Too many branches,Too many arguments in function definition
        self,
        network,
        graphics,
        *,
        url=None,
        headers=None,
        json_path=None,
        regexp_path=None,
        json_transform=None,
        success_callback=None,
        debug=False,
    ):
        self.network = network
        """The :py:class:`~adafruit_portalbase.NetworkBase`-derived instance provided"""
        self.graphics = graphics
        """The :py:meth:`displayio.Group()` object that acts as the root group screen
        for this device."""

        # Font Cache
        self._fonts = {}
        self._text = []

        try:
            import alarm

            self._alarm = alarm
        except ImportError:
            self._alarm = None
        self._debug = debug
        if url and self.network is None:
            raise RuntimeError("network must not be None to get data from a url")
        self.url = url
        if headers and self.network is None:
            raise RuntimeError("network must not be None to send headers")
        self._headers = headers
        if json_path and self.network is None:
            raise RuntimeError("network must not be None to use json_path")
        self._json_path = None
        self.json_path = json_path

        if regexp_path and self.network is None:
            raise RuntimeError("network must not be None to use regexp_path")
        self._regexp_path = regexp_path

        if success_callback and self.network is None:
            raise RuntimeError("network must not be None to use success_callback")
        self._success_callback = success_callback

        # Add any JSON translators

        if json_transform:
            if self.network is not None:
                self.network.add_json_transform(json_transform)
            else:
                raise RuntimeError("network must not be None to use json_transform.")

    def _load_font(self, font):
        """
        Load and cache a font if not previously loaded
        Return the key of the cached font

        :param font: Either terminalio.FONT or the path to the bdf font file

        """
        if font is terminalio.FONT:
            if "terminal" not in self._fonts:
                self._fonts["terminal"] = terminalio.FONT
            return "terminal"
        if font not in self._fonts:
            self._fonts[font] = bitmap_font.load_font(font)
        return font

    @staticmethod
    def html_color_convert(color):
        """Convert an HTML color code to an integer

        :param color: The color value to be converted

        """
        if isinstance(color, str):
            if color[0] == "#":
                color = color.lstrip("#")
            return int(color, 16)
        return color  # Return unconverted

    @staticmethod
    def wrap_nicely(string, max_chars):
        """A helper that will return a list of lines with word-break wrapping.

        :param str string: The text to be wrapped.
        :param int max_chars: The maximum number of characters on a line before wrapping.

        """
        return wrap_text_to_lines(string, max_chars)

    def add_text(  # noqa: PLR0913 Too many arguments in function definition
        self,
        text_position=(0, 0),
        text_font=terminalio.FONT,
        text_color=0x000000,
        text_wrap=0,
        text_maxlen=0,
        text_transform=None,
        text_scale=1,
        line_spacing=1.25,
        text_anchor_point=(0, 0.5),
        outline_size=0,
        outline_color=0x000000,
        is_data=True,
        text=None,
    ) -> int:
        """
        Add text labels with settings. Returns the index of the label,
        for use with ``set_text()`` and ``set_text_color()``.

        :param str text_font: The path to your font file for your data text display.
        :param text_position: The position of your extracted text on the display in an (x, y) tuple.
                              Can be a list of tuples for when there's a list of json_paths, for
                              example.
        :param text_color: The color of the text, in 0xRRGGBB format. Can be a list of colors for
                           when there's multiple texts. Defaults to ``None``.
        :param text_wrap: When non-zero, the maximum number of characters on each line before text
                          is wrapped. (for long text data chunks). Defaults to 0, no wrapping.
        :param text_maxlen: The max length of the text. If non-zero, it will be truncated to this
                            length. Defaults to 0.
        :param text_transform: A function that will be called on the text before display
        :param int text_scale: The factor to scale the default size of the text by
        :param float line_spacing: The factor to space the lines apart
        :param (float,float) text_anchor_point: Values between 0 and 1 to indicate where the text
                                                 position is relative to the label
        :param bool is_data: If True, fetch will attempt to update the label
        :param str text: If this is provided, it will set the initial text of the label.

        :return: Index of the new text label.
        :rtype: int

        """
        if not text_wrap:
            text_wrap = 0
        if not text_maxlen:
            text_maxlen = 0
        if not text_transform:
            text_transform = None
        if not isinstance(text_scale, (int, float)) or text_scale < 1:
            text_scale = 1
        if not isinstance(text_anchor_point, (tuple, list)):
            text_anchor_point = (0, 0.5)
        if not 0 <= text_anchor_point[0] <= 1 or not 0 <= text_anchor_point[1] <= 1:
            raise ValueError("Text anchor point values should be between 0 and 1.")
        text_scale = round(text_scale)
        gc.collect()

        if self._debug:
            print("Init text area")
        text_field = {
            "label": None,
            "font": self._load_font(text_font),
            "color": self.html_color_convert(text_color),
            "position": text_position,
            "wrap": text_wrap,
            "maxlen": text_maxlen,
            "transform": text_transform,
            "scale": text_scale,
            "line_spacing": line_spacing,
            "anchor_point": text_anchor_point,
            "is_data": bool(is_data),
            "outline_size": outline_size,
            "outline_color": outline_color,
        }
        self._text.append(text_field)

        text_index = len(self._text) - 1
        if text is not None:
            self.set_text(text, text_index)

        return text_index

    def remove_all_text(self, clear_font_cache=False):
        """Remove all added text and labels.

        :param bool clear_font_cache: Clear the font cache. Defaults to False.
        """

        # Remove the labels
        for i in range(len(self._text)):
            self.set_text("", i)
        # Remove the data
        self._text = []
        if clear_font_cache:
            self._fonts = {}
        gc.collect()

    def set_text(self, val, index=0):  # noqa: PLR0912 Too many branches
        """Display text, with indexing into our list of text boxes.

        :param str val: The text to be displayed
        :param index: Defaults to 0.

        """
        # Make sure at least a single label exists
        if not self._text:
            self.add_text()
        string = str(val)
        if self._text[index]["maxlen"] and len(string) > self._text[index]["maxlen"]:
            # too long! shorten it
            if len(string) >= 3:
                string = string[: self._text[index]["maxlen"] - 3] + "..."
            else:
                string = string[: self._text[index]["maxlen"]]
        index_in_root_group = None

        if len(string) > 0 and self._text[index]["wrap"]:
            if self._debug:
                print("Wrapping text with length of", self._text[index]["wrap"])
            lines = self.wrap_nicely(string, self._text[index]["wrap"])
            string = "\n".join(lines)

        if self._text[index]["label"] is not None:
            if self._debug:
                print("Replacing text area with :", string)
            index_in_root_group = self.root_group.index(self._text[index]["label"])
        elif self._debug:
            print("Creating text area with :", string)
        if len(string) > 0:
            if self._text[index]["label"] is None:
                if self._text[index]["outline_size"] == 0:
                    self._text[index]["label"] = Label(
                        self._fonts[self._text[index]["font"]],
                        text=string,
                        scale=self._text[index]["scale"],
                    )
                else:
                    self._text[index]["label"] = OutlinedLabel(
                        self._fonts[self._text[index]["font"]],
                        text=string,
                        scale=self._text[index]["scale"],
                        outline_size=self._text[index]["outline_size"],
                        outline_color=self._text[index]["outline_color"],
                    )
                if index_in_root_group is not None:
                    self.root_group[index_in_root_group] = self._text[index]["label"]
                else:
                    self.root_group.append(self._text[index]["label"])
            else:
                self._text[index]["label"].text = string
            self._text[index]["label"].color = self._text[index]["color"]
            self._text[index]["label"].anchor_point = self._text[index]["anchor_point"]
            self._text[index]["label"].anchored_position = self._text[index]["position"]
            self._text[index]["label"].line_spacing = self._text[index]["line_spacing"]
        elif index_in_root_group is not None:
            self._text[index]["label"] = None

        # Remove the label from root group
        if index_in_root_group is not None and self._text[index]["label"] is None:
            del self.root_group[index_in_root_group]
        gc.collect()

    def preload_font(self, glyphs=None, index=0):
        """Preload font.

        :param glyphs: The font glyphs to load. Defaults to ``None``, uses alphanumeric glyphs if
                       None.
        """
        if not glyphs:
            glyphs = b"0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ-!,. \"'?!"
        print("Preloading font glyphs:", glyphs)
        if self._fonts[self._text[index]["font"]] is not terminalio.FONT:
            self._fonts[self._text[index]["font"]].load_glyphs(glyphs)

    def set_headers(self, headers):
        """Set the headers used by fetch().

        :param headers: The new header dictionary

        """
        self._headers = headers

    def set_background(self, file_or_color, position=None):
        """The background image to a bitmap file.

        :param file_or_color: The filename of the chosen background image, or a hex color.

        """
        self.graphics.set_background(file_or_color, position)

    def set_text_color(self, color, index=0):
        """Update the text color, with indexing into our list of text boxes.

        :param int color: The color value to be used
        :param index: Defaults to 0.

        """
        if self._text[index]:
            color = self.html_color_convert(color)
            self._text[index]["color"] = color
            if self._text[index]["label"] is not None:
                self._text[index]["label"].color = color

    def create_time_alarm(self, sleep_time):
        """
        Create a TimeAlarm based on the specified amount of delay

        :param float sleep_time: The amount of time to sleep in seconds

        """
        if self._alarm:
            return self._alarm.time.TimeAlarm(monotonic_time=time.monotonic() + sleep_time)
        raise NotImplementedError(
            "Alarms not supported. Make sure you have the latest CircuitPython."
        )

    def create_pin_alarm(self, pin, value, edge=False, pull=False):
        """
        Create a PinAlarm that is triggered when the pin has a specific value

        :param microcontroller.Pin pin: The trigger pin.
        :param bool value: The value on which to trigger.
        :param bool edge:  Trigger only when there is a transition.
        :param bool pull: Enable a pull-up or pull-down for the ``pin``.

        """
        if self._alarm:
            return self._alarm.pin.PinAlarm(pin, value, edge, pull)
        raise NotImplementedError(
            "Alarms not supported. Make sure you have the latest CircuitPython."
        )

    def create_touch_alarm(self, pin):
        """
        Create a TouchAlarm that is triggered when the pin is touched.

        :param microcontroller.Pin pin: The trigger pin.
        """
        if self._alarm:
            return self._alarm.touch.TouchAlarm(pin)
        raise NotImplementedError(
            "Alarms not supported. Make sure you have the latest CircuitPython."
        )

    def exit_and_deep_sleep(self, alarms):
        """
        Stops the current program and enters deep sleep. The program is restarted from the beginning
        after the alarm or alarms are triggered.

        See https://circuitpython.readthedocs.io/en/latest/shared-bindings/alarm/index.html for more
        details.

        :param float alarms: The alarm or alarms to use as a trigger

        """

        # For backwards compatibility
        if isinstance(alarms, (float, int)):
            alarms = self.create_time_alarm(alarms)

        self._alarm.exit_and_deep_sleep_until_alarms(alarms)

    def enter_light_sleep(self, alarms):
        """
        Enter light sleep and resume the program after a certain period of time.

        See https://circuitpython.readthedocs.io/en/latest/shared-bindings/alarm/index.html for more
        details.

        :param float sleep_time: The amount of time to sleep in seconds

        """
        # For backwards compatibility
        if isinstance(alarms, (float, int)):
            alarms = self.create_time_alarm(alarms)

        self._alarm.light_sleep_until_alarms(alarms)

    def _fetch_set_text(self, val, index=0):
        self.set_text(val, index=index)

    def fetch(self, refresh_url=None, timeout=10):
        """Fetch data from the url we initialized with, perfom any parsing,
        and display text or graphics. This function does pretty much everything
        Optionally update the URL

        :param str refresh_url: The overriding URL to fetch from. Defaults to ``None``.
        :param int timeout: The timeout period in seconds.

        """

        if self.network is None:
            raise RuntimeError("network must not be None to use fetch()")
        if refresh_url:
            self.url = refresh_url
        values = []

        values = self.network.fetch_data(
            self.url,
            headers=self._headers,
            json_path=self._json_path,
            regexp_path=self._regexp_path,
            timeout=timeout,
        )

        # if we have a callback registered, call it now
        if self._success_callback:
            self._success_callback(values)

        self._fill_text_labels(values)

        if len(values) == 1:
            return values[0]
        return values

    def _fill_text_labels(self, values):
        # fill out all the text blocks
        if self._text:
            value_index = 0  # In case values and text is not the same
            for i in range(len(self._text)):
                if (not self._text[i]["is_data"]) or (value_index > (len(values) - 1)):
                    continue
                string = None
                if self._text[i]["transform"]:
                    func = self._text[i]["transform"]
                    string = func(values[value_index])
                else:
                    try:
                        string = f"{int(values[value_index]):,d}"
                    except (TypeError, ValueError):
                        string = values[value_index]  # ok it's a string
                self._fetch_set_text(string, index=i)
                value_index += 1

    def get_local_time(self, location=None, max_attempts=10):
        """Accessor function for get_local_time()"""
        if self.network is None:
            raise RuntimeError("network must not be None to use get_local_time()")

        return self.network.get_local_time(location=location, max_attempts=max_attempts)

    def push_to_io(self, feed_key, data, metadata=None, precision=None):
        """Push data to an adafruit.io feed

        :param str feed_key: Name of feed key to push data to.
        :param data: data to send to feed
        :param dict metadata: Optional metadata associated with the data
        :param int precision: Optional amount of precision points to send with floating point data

        """
        if self.network is None:
            raise RuntimeError("network must not be None to use push_to_io()")

        self.network.push_to_io(feed_key, data, metadata=metadata, precision=precision)

    def get_io_data(self, feed_key):
        """Return all values from the Adafruit IO Feed Data that matches the feed key

        :param str feed_key: Name of feed key to receive data from.

        """
        if self.network is None:
            raise RuntimeError("network must not be None to use get_io_data()")

        return self.network.get_io_data(feed_key)

    def get_io_feed(self, feed_key, detailed=False):
        """Return the Adafruit IO Feed that matches the feed key

        :param str feed_key: Name of feed key to match.
        :param bool detailed: Whether to return additional detailed information

        """
        if self.network is None:
            raise RuntimeError("network must not be None to use get_io_feed()")

        return self.network.get_io_feed(feed_key, detailed)

    def get_io_group(self, group_key):
        """Return the Adafruit IO Group that matches the group key

        :param str group_key: Name of group key to match.

        """
        if self.network is None:
            raise RuntimeError("network must not be None to use get_io_group()")
        return self.network.get_io_group(group_key)

    @property
    def json_path(self):
        """
        Get or set the list of json traversal to get data out of. Can be list
        of lists for multiple data points.
        """
        return self._json_path

    @json_path.setter
    def json_path(self, value):
        if value is not None and self.network is None:
            raise RuntimeError("network must not be None to use json_path.")

        if value:
            if isinstance(value[0], (list, tuple)):
                self._json_path = value
            else:
                self._json_path = (value,)
        else:
            self._json_path = None

    @property
    def root_group(self):
        """The root display group for this device."""
        return self.graphics.root_group

    @property
    def splash(self):
        """The root display group for this device (for backwards compatibility)."""
        print(
            "WARNING: splash is deprecated, use root_group instead. "
            "This will be removed in a future release."
        )
        return self.graphics.root_group

    @property
    def display(self):
        """The displayio.Display object for this device."""
        return self.graphics.display

    @property
    def text_fields(self):
        """
        The list of text field(s) metadata objects.
        See add_text() definition for available metadata fields.
        """
        return self._text
